using Microsoft.Xna.Framework;
using Microsoft.Xna.Framework.Graphics;
using Microsoft.Xna.Framework.Input;

namespace ColourBlindness
{
    /// <summary>
    /// Responsibl for simulating various forms of colour blindness.
    /// </summary>
    public class ColourBlindnessSimulator : Game
    {
        private readonly GraphicsDeviceManager _graphics;
        private SpriteBatch _spriteBatch;
        private SpriteFont _spriteFont;
        private Effect _effect;

        /// <summary>The texture used to display our colour map</summary>
        /// <remarks>Colour range from #000000 to #ffffff in steps of #000033.</remarks>
        private Texture2D _texture;

        /// <summary>The index to the current colour filter.</summary>
        private int _filterIndex;

        /// <summary>An array containing descriptions for the available filters.</summary>
        private readonly string[] _filters = new [] {"1 - Normal Vision", "2 - Protanopia (no red)", "3 - Deuteranpoia (no green)", "4 - Tritanopia (no blue)"};

        /// <summary>
        /// Initialises a new instance of a <see cref="ColourBlindnessSimulator">ColourBlindnessSimulator</see>.
        /// </summary>
        public ColourBlindnessSimulator()
        {
            this._graphics = new GraphicsDeviceManager(this);
            Content.RootDirectory = "Content";

            this._graphics.PreferredBackBufferWidth = 800;
            this._graphics.PreferredBackBufferHeight = 760;
        }

        /// <summary>
        /// LoadContent will be called once per game and is the place to load all of your content.
        /// </summary>
        protected override void LoadContent()
        {
            // Create a new SpriteBatch, which can be used to draw textures.
            this._spriteBatch = new SpriteBatch(this.GraphicsDevice);

            this._effect = this.Content.Load<Effect>("Shaders\\ColourBlind");
            this._effect.Parameters["Viewport"].SetValue(new Vector2(this.GraphicsDevice.Viewport.Width, this.GraphicsDevice.Viewport.Height));

            this._texture = this.Content.Load<Texture2D>("Textures\\Colours");
            this._spriteFont = this.Content.Load<SpriteFont>("Fonts\\Font");
        }

        /// <summary>
        /// Allows the game to run logic such as updating the world, checking for collisions, gathering input, and playing audio.
        /// </summary>
        /// <param name="gameTime">Provides a snapshot of timing values.</param>
        protected override void Update(GameTime gameTime)
        {
            // Allows the game to exit
            if (GamePad.GetState(PlayerIndex.One).Buttons.Back == ButtonState.Pressed)
                this.Exit();

            //Monitor the keyboard - change filter types when the following keys are down: 1, 2, 3, 4.
            KeyboardState keyState = Keyboard.GetState();
            if (keyState.IsKeyDown(Keys.D1))
            {
                this._filterIndex = 0;
                this._effect.Parameters["FilterIndex"].SetValue(this._filterIndex);
            }
            else if (keyState.IsKeyDown(Keys.D2))
            {
                this._filterIndex = 1;
                this._effect.Parameters["FilterIndex"].SetValue(this._filterIndex);
            }
            else if (keyState.IsKeyDown(Keys.D3))
            {
                this._filterIndex = 2;
                this._effect.Parameters["FilterIndex"].SetValue(this._filterIndex);
            }
            else if (keyState.IsKeyDown(Keys.D4))
            {
                this._filterIndex = 3;
                this._effect.Parameters["FilterIndex"].SetValue(this._filterIndex);
            }

            base.Update(gameTime);
        }

        /// <summary>
        /// This is called when the game should draw itself.
        /// </summary>
        /// <param name="gameTime">Provides a snapshot of timing values.</param>
        protected override void Draw(GameTime gameTime)
        {
            this.GraphicsDevice.Clear(Color.Black);

            // Draw the texture
            this._spriteBatch.Begin(SpriteSortMode.Immediate, BlendState.AlphaBlend, null, null, null, _effect);
            this._spriteBatch.Draw(this._texture, new Vector2(50.0f), Color.White);
            this._spriteBatch.End();

            // Draw the helper text
            this._spriteBatch.Begin();
            this._spriteBatch.DrawString(this._spriteFont, this._filters[this._filterIndex], new Vector2(50.0f, 540.0f), Color.White, 0.0f, Vector2.Zero, 1.5f, SpriteEffects.None, 0.0f);
            this._spriteBatch.DrawString(this._spriteFont, this._filters[0], new Vector2(50.0f, 600.0f), Color.LightGray);
            this._spriteBatch.DrawString(this._spriteFont, this._filters[1], new Vector2(50.0f, 630.0f), Color.LightGray);
            this._spriteBatch.DrawString(this._spriteFont, this._filters[2], new Vector2(50.0f, 660.0f), Color.LightGray);
            this._spriteBatch.DrawString(this._spriteFont, this._filters[3], new Vector2(50.0f, 690.0f), Color.LightGray);
            this._spriteBatch.End();

            base.Draw(gameTime);
        }
    }
}
