//------------------------------------------------------------------------------------------------------------------------------------------------------
// Description: Shader to simulate various forms of colour blindness
// Author:		Paul Cunningham
// Date:		24/11/2010
// Sources:		This shader implementation is based on the the Display Filter, "cdisplay_colorblind.c" for GIMP
//				http://gimp.sourcearchive.com/lines/2.6.6-2/display-filter-color-blind_8c-source.html
//------------------------------------------------------------------------------------------------------------------------------------------------------

//------------------------------------------------------------------------------------------------------------------------------------------------------
// For most modern Cathode-Ray Tube monitors (CRTs), the following are good estimates of the RGB->LMS and LMS->RGB transform matrices.  They are 
// based on spectra measured on a typical CRT with a PhotoResearch PR650 spectral photometer and the Stockma human cone fundamentals. 
// NOTE: these estimates will NOT work well for LCDs!
//------------------------------------------------------------------------------------------------------------------------------------------------------
static const float rgbToLms[9] =
{
	0.05059983, 
	0.08585369,
	0.00952420,
 
	0.01893033,
	0.08925308,
	0.01370054,
 
	0.00292202,
	0.00975732,
	0.07145979
};
 
static const float lmsToRgb[9] =
{
	30.830854,
	-29.832659,
	1.610474,
 
	-6.481468,
	17.715578,
	-2.532642,
 
    -0.375690,
	-1.199062,
	14.273846
};



//------------------------------------------------------------------------------------------------------------------------------------------------------
// Load the LMS anchor-point values for lambda = 475 & 485 nm (for protans & deutans) and the LMS values for lambda = 575 & 660 nm (for tritans)
//------------------------------------------------------------------------------------------------------------------------------------------------------
static const float anchor[12] =
{
	0.08008, 0.1579, 0.5897,
	0.1284, 0.2237, 0.3636,
	0.9856, 0.7325, 0.001079,
	0.0914, 0.007009, 0.0
};

static const float anchor_e[3] =
{
	rgbToLms[0] + rgbToLms[1] + rgbToLms[2],
	rgbToLms[3] + rgbToLms[4] + rgbToLms[5],
	rgbToLms[6] + rgbToLms[7] + rgbToLms[8],
};



//------------------------------------------------------------------------------------------------------------------------------------------------------
// We also need LMS for RGB=(1,1,1)- the equal-energy point (one of our anchors) (we can just peel this out of the rgb2lms transform matrix)
//------------------------------------------------------------------------------------------------------------------------------------------------------



//------------------------------------------------------------------------------------------------------------------------------------------------------
// Parameters 
//------------------------------------------------------------------------------------------------------------------------------------------------------

//------------------------------------------------------------------------------------------------------------------------------------------------------
// FilterIndex
//------------------------------------------------------------------------------------------------------------------------------------------------------
// 0 NORMAL VISION
// 1 PROTANOPIA (no red)
// 2 DEUTERANOPIA (no green) 
// 3 TRITANOPIA (no blue)

int FilterIndex = 0; 

//------------------------------------------------------------------------------------------------------------------------------------------------------
// GammaRGB
//------------------------------------------------------------------------------------------------------------------------------------------------------ 
// The RGB<->LMS transforms above are computed from the human cone photo-pigment absorption spectra and the monitor phosphor
// emission spectra. These parameters are fairly constant for most humans and most montiors (at least for modern CRTs). However,
// gamma will vary quite a bit, as it is a property of the monitor (eg. amplifier gain), the video card, and even the
// software. Further, users can adjust their gammas (either via adjusting the monitor amp gains or in software). That said, the
// following are the gamma estimates that were used in the Vischeck code. Many colorblind users viewed their simulations 
// and told them that they "work" (simulated and original images are indistinguishabled).

float GammaRGB = 2.1f;

float2 Viewport;

sampler2D baseSampler: register(s0);

struct ColourBlind
{
	float a1;
	float b1;
	float c1;
	float a2;
	float b2;
	float c2;
	float inflection;
};



//------------------------------------------------------------------------------------------------------------------------------------------------------
// Set up the arrays for the different types of deficiency
//------------------------------------------------------------------------------------------------------------------------------------------------------
ColourBlind deficiencyAnchor[3] = 
{
    // PROTANOPIA (no red)
    // Set a, b, c for lam = 575nm and lam = 475
	{
		anchor_e[1] * anchor[8] - anchor_e[2] * anchor[7],
		anchor_e[2] * anchor[6] - anchor_e[0] * anchor[8],
		anchor_e[0] * anchor[7] - anchor_e[1] * anchor[6],
		anchor_e[1] * anchor[2] - anchor_e[2] * anchor[1],
		anchor_e[2] * anchor[0] - anchor_e[0] * anchor[2],
		anchor_e[0] * anchor[1] - anchor_e[1] * anchor[0],
		anchor_e[2] / anchor_e[1]
    },

	// DEUTERANOPIA (no green)
	// Set a, b, c for lam = 575nm and lam = 475 
	{
		anchor_e[1] * anchor[8] - anchor_e[2] * anchor[7],
		anchor_e[2] * anchor[6] - anchor_e[0] * anchor[8],
		anchor_e[0] * anchor[7] - anchor_e[1] * anchor[6],
		anchor_e[1] * anchor[2] - anchor_e[2] * anchor[1],
		anchor_e[2] * anchor[0] - anchor_e[0] * anchor[2],
		anchor_e[0] * anchor[1] - anchor_e[1] * anchor[0],
		anchor_e[2] / anchor_e[0]
	},
 
    // TRITANOPIA (no blue)
	// Set a, b, c, for lam=575nm and lam=475
	{
       anchor_e[1] * anchor[11] - anchor_e[2] * anchor[10],
       anchor_e[2] * anchor[9]  - anchor_e[0] * anchor[11],
       anchor_e[0] * anchor[10] - anchor_e[1] * anchor[9],
       anchor_e[1] * anchor[5]  - anchor_e[2] * anchor[4],
       anchor_e[2] * anchor[3]  - anchor_e[0] * anchor[5],
       anchor_e[0] * anchor[4]  - anchor_e[1] * anchor[3],
       anchor_e[1] / anchor_e[0]
    }
};



//------------------------------------------------------------------------------------------------------------------------------------------------------
// A simple vertex shader
//------------------------------------------------------------------------------------------------------------------------------------------------------
void SimpleVS(inout float4 color : COLOR0, inout float2 texCoord : TEXCOORD0, inout float4 position : POSITION0)
{
   // Half pixel offset for correct texel centering.
   position.xy -= 0.5;

   // Viewport adjustment.
   position.xy = position.xy / Viewport;
   position.xy *= float2(2, -2);
   position.xy -= float2(1, -1);
}



//------------------------------------------------------------------------------------------------------------------------------------------------------
// Pixel Shader to simulate various forms of colour blindness
//------------------------------------------------------------------------------------------------------------------------------------------------------
float4 ColourBlindPS(float4 color : COLOR0, float2 texCoord : TEXCOORD0, float4 position : POSITION0) : COLOR0
{
	//Get the pixels colour from the texture.
    float4 output = tex2D(baseSampler, texCoord);
                
	//Apply tinting from the Color value passed to the SpriteBatch
	output *= color;

	//Start of colour blindness simulator

	//Return the normal colour if no deficiency
	if (FilterIndex <= 0)
		return output;

	float deficiencyIndex = FilterIndex - 1;
	ColourBlind deficiency = deficiencyAnchor[deficiencyIndex];
	 
	//Remove gamma to linearize RGB intensities (note convert 0.0f <--> 1.0f to 0 <-> 255 before doing so)
	float scale = 1.0 / GammaRGB;
	float red = pow(output.r * 255, scale);
	float green = pow(output.g * 255, scale);
	float blue = pow(output.b * 255, scale); 

	// Convert to LMS (dot product with transform matrix) 
	float redOld   = red;
	float greenOld = green;

	red = redOld * rgbToLms[0] + greenOld * rgbToLms[1] + blue * rgbToLms[2];
	green = redOld * rgbToLms[3] + greenOld * rgbToLms[4] + blue * rgbToLms[5];
	blue  = redOld * rgbToLms[6] + greenOld * rgbToLms[7] + blue * rgbToLms[8];

	float tmp;	
	// PROTANOPIA (no red)
	if (deficiencyIndex == 0)
    {       
		tmp = blue / green;
        // See which side of the inflection line we fall...
        if (tmp < deficiency.inflection)
			red = -(deficiency.b1 * green + deficiency.c1 * blue) / deficiency.a1;
        else
            red = -(deficiency.b2 * green + deficiency.c2 * blue) / deficiency.a2;
	}
	// DEUTERANOPIA (no green)
	else if (deficiencyIndex == 1)
	{
		tmp = blue / red;
		// See which side of the inflection line we fall...
		if (tmp < deficiency.inflection)
			green = -(deficiency.a1 * red + deficiency.c1 * blue) / deficiency.b1;
        else
            green = -(deficiency.a2 * red + deficiency.c2 * blue) / deficiency.b2;
	}
	// TRITANOPIA (no blue)
	else
	{
        tmp = green / red;
        // See which side of the inflection line we fall...
        if (tmp < deficiency.inflection)
			blue = -(deficiency.a1 * red + deficiency.b1 * green) / deficiency.c1;
        else
            blue = -(deficiency.a2 * red + deficiency.b2 * green) / deficiency.c2;
    }
 
    // Convert back to RGB (cross product with transform matrix) 
    redOld   = red;
    greenOld = green;
 
    red = redOld * lmsToRgb[0] + greenOld * lmsToRgb[1] + blue * lmsToRgb[2];
    green = redOld * lmsToRgb[3] + greenOld * lmsToRgb[4] + blue * lmsToRgb[5];
    blue = redOld * lmsToRgb[6] + greenOld * lmsToRgb[7] + blue * lmsToRgb[8];
 
	// Apply gamma to go back to non-linear intensities
	//(note convert 0 <-> 255 to 0.0f <--> 1.0f after doing so)
	output.r = pow(red, GammaRGB) / 255.0f;
	output.g = pow(green, GammaRGB) / 255.0f;
	output.b = pow(blue, GammaRGB) / 255.0f;

    return output;
}



//------------------------------------------------------------------------------------------------------------------------------------------------------
// Technique Section
//------------------------------------------------------------------------------------------------------------------------------------------------------
technique Simple
{
   pass Single_Pass
   {	
        VertexShader = compile vs_3_0 SimpleVS();
        PixelShader = compile ps_3_0 ColourBlindPS();    
   }
}